#include <iostream>
#include <vector>
#include <map>
#include <algorithm>

using namespace std;

const long long MOD = 1000000007;

struct Graph {
    int n;
    vector<pair<int, int>> edges;

    Graph(int n) : n(n) {}

    void add_edge(int u, int v) {
        edges.push_back({min(u, v), max(u, v)});
    }

    bool operator<(const Graph& other) const {
        if (n != other.n) return n < other.n;
        return edges < other.edges;
    }
};

map<Graph, vector<long long>> memo;

// Compute chromatic polynomial coefficients
// Returns vector where result[i] is coefficient of k^i
vector<long long> chromatic_poly(Graph g) {
    // Base case: no vertices
    if (g.n == 0) {
        return {1};
    }

    // Base case: no edges - answer is k^n
    if (g.edges.empty()) {
        vector<long long> result(g.n + 1, 0);
        result[g.n] = 1;
        return result;
    }

    // Check memo
    if (memo.count(g)) {
        return memo[g];
    }

    // Normalize edges and remove duplicates
    sort(g.edges.begin(), g.edges.end());
    g.edges.erase(unique(g.edges.begin(), g.edges.end()), g.edges.end());

    // Pick first edge to delete/contract
    auto edge = g.edges[0];
    int u = edge.first, v = edge.second;

    // Graph with edge deleted
    Graph g_delete = g;
    g_delete.edges.erase(g_delete.edges.begin());

    // Graph with edge contracted
    Graph g_contract(g.n - 1);
    // Map v to u, and shift vertices > v down by 1
    for (size_t i = 1; i < g.edges.size(); i++) {
        int a = g.edges[i].first;
        int b = g.edges[i].second;

        // Replace v with u
        if (a == v) a = u;
        if (b == v) b = u;

        // Skip self-loops
        if (a == b) continue;

        // Shift down vertices > v
        if (a > v) a--;
        if (b > v) b--;

        g_contract.add_edge(a, b);
    }

    // Deletion-contraction: P(G) = P(G-e) - P(G/e)
    auto poly_delete = chromatic_poly(g_delete);
    auto poly_contract = chromatic_poly(g_contract);

    // Subtract: poly_delete - poly_contract
    int max_deg = max(poly_delete.size(), poly_contract.size());
    vector<long long> result(max_deg, 0);

    for (size_t i = 0; i < poly_delete.size(); i++) {
        result[i] = poly_delete[i];
    }
    for (size_t i = 0; i < poly_contract.size(); i++) {
        result[i] = (result[i] - poly_contract[i] + MOD) % MOD;
    }

    memo[g] = result;
    return result;
}

long long eval_poly(const vector<long long>& poly, long long k) {
    long long result = 0;
    long long k_pow = 1;

    for (long long coef : poly) {
        result = (result + coef * k_pow) % MOD;
        k_pow = (k_pow * k) % MOD;
    }

    return result;
}

int main() {
    int n, m, t;
    cin >> n >> m >> t;

    vector<int> queries(t);
    for (int i = 0; i < t; i++) {
        cin >> queries[i];
    }

    Graph g(n);
    for (int i = 0; i < m; i++) {
        int u, v;
        cin >> u >> v;
        g.add_edge(u, v);
    }

    // Compute chromatic polynomial
    vector<long long> poly = chromatic_poly(g);

    // Answer queries
    for (int k : queries) {
        cout << eval_poly(poly, k) << "\n";
    }

    return 0;
}
