#include <iostream>
#include <vector>
#include <numeric>
#include <map>
#include <algorithm>
#include <set>

// Define the modulus for all calculations
const int MOD = 1000000007;

// Memoization table.
// Key: A pair of (number of vertices, sorted edge list) uniquely representing a graph.
// Value: The chromatic polynomial as a map from a power to its coefficient.
std::map<std::pair<int, std::vector<std::pair<int, int>>>, std::map<int, long long>> memo;

// Forward declaration of the recursive function.
std::map<int, long long> compute_poly(int n, std::vector<std::pair<int, int>> edges);

/**
 * @brief Computes (base^exp) % MOD using binary exponentiation.
 * This is an efficient way to calculate large powers under a modulus.
 * @param base The base of the power.
 * @param exp The exponent.
 * @return The result of (base^exp) % MOD.
 */
long long power(long long base, long long exp) {
    long long res = 1;
    base %= MOD;
    while (exp > 0) {
        if (exp % 2 == 1) res = (res * base) % MOD;
        base = (base * base) % MOD;
        exp /= 2;
    }
    return res;
}

/**
 * @brief Recursively computes the chromatic polynomial of a graph G = (V, E)
 * using the deletion-contraction formula: P(G, k) = P(G-e, k) - P(G/e, k).
 * @param n The number of vertices in the graph.
 * @param edges The list of edges in the graph.
 * @return The chromatic polynomial as a map from power to coefficient.
 */
std::map<int, long long> compute_poly(int n, std::vector<std::pair<int, int>> edges) {
    // To make memoization effective, we need a canonical representation for the graph.
    // We ensure that for each edge (u, v), u < v, and then sort the entire edge list.
    for (auto& edge : edges) {
        if (edge.first > edge.second) {
            std::swap(edge.first, edge.second);
        }
    }
    std::sort(edges.begin(), edges.end());
    std::pair<int, std::vector<std::pair<int, int>>> state = {n, edges};

    // 1. Check memoization table to see if we've solved this subproblem before.
    if (memo.count(state)) {
        return memo[state];
    }

    // 2. Base Case: If there are no edges, the graph is a set of isolated vertices.
    // The number of ways to color it is k^n (k choices for each of the n vertices).
    // The polynomial is therefore just the single term 1 * k^n.
    if (edges.empty()) {
        std::map<int, long long> poly;
        poly[n] = 1;
        return memo[state] = poly;
    }

    // 3. Recursive Step: Pick an edge e = (u, v) to apply the formula on.
    std::pair<int, int> e = edges.back();
    edges.pop_back();
    int u = e.first;
    int v = e.second;

    // 3a. Compute polynomial for G-e (Deletion)
    // The graph G-e has the same 'n' vertices but one fewer edge.
    std::map<int, long long> poly_deleted = compute_poly(n, edges);

    // 3b. Compute polynomial for G/e (Contraction)
    // In G/e, we merge vertex v into u. The number of vertices becomes n-1.
    // To keep the vertex labels dense (from 0 to n-2), we must relabel all vertices w > v to w-1.
    int contracted_n = n - 1;
    std::vector<std::pair<int, int>> contracted_edges;
    for (const auto& edge : edges) {
        int n1 = edge.first;
        int n2 = edge.second;

        // Remap any endpoint that was v to u
        if (n1 == v) n1 = u;
        if (n2 == v) n2 = u;

        // Relabel vertices that were > v to maintain a dense set {0, ..., n-2}
        if (n1 > v) n1--;
        if (n2 > v) n2--;

        // After remapping, avoid adding self-loops (e.g., u-u) which might be created.
        if (n1 != n2) {
            contracted_edges.push_back({n1, n2});
        }
    }

    // The contracted graph might now have parallel edges. We must remove them
    // to keep the graph simple, as the chromatic polynomial is for simple graphs.
    // Sorting and using std::unique is an effective way to do this.
    for (auto& edge : contracted_edges) {
        if (edge.first > edge.second) std::swap(edge.first, edge.second);
    }
    std::sort(contracted_edges.begin(), contracted_edges.end());
    contracted_edges.erase(std::unique(contracted_edges.begin(), contracted_edges.end()), contracted_edges.end());

    std::map<int, long long> poly_contracted = compute_poly(contracted_n, contracted_edges);

    // 4. Combine results: P(G,k) = P(G-e,k) - P(G/e,k)
    // This translates to subtracting the coefficients of the resulting polynomials.
    std::map<int, long long> result_poly = poly_deleted;
    for (auto const& [power, coeff] : poly_contracted) {
        result_poly[power] -= coeff;
    }

    // 5. Memoize the result for the current state and return it.
    return memo[state] = result_poly;
}


int main() {
    // Fast I/O
    std::ios_base::sync_with_stdio(false);
    std::cin.tie(NULL);

    int n, m, t;
    std::cin >> n >> m >> t;

    std::vector<long long> a(t);
    for (int i = 0; i < t; ++i) {
        std::cin >> a[i];
    }

    std::vector<std::pair<int, int>> initial_edges(m);
    for (int i = 0; i < m; ++i) {
        std::cin >> initial_edges[i].first >> initial_edges[i].second;
    }

    // Compute the chromatic polynomial for the initial graph using the recursive function.
    std::map<int, long long> final_poly = compute_poly(n, initial_edges);

    // Process each query by evaluating the computed polynomial for the given k=a[i].
    for (int i = 0; i < t; ++i) {
        long long k = a[i];
        long long result = 0;

        // Evaluate P_G(k) = sum(c_p * k^p)
        for (auto const& [p, c] : final_poly) {
            if (c == 0) continue;

            // Handle potential negative coefficients correctly with modulo arithmetic.
            long long coeff_mod = (c % MOD + MOD) % MOD;
            long long k_pow = power(k, p);
            long long term = (coeff_mod * k_pow) % MOD;
            result = (result + term) % MOD;
        }

        // Print each result on a new line.
        std::cout << result << "\n";
    }

    return 0;
}
