#include <iostream>
#include <vector>
#include <numeric>

// Define the modulus for all calculations
const int MOD = 1000000007;

/**
 * @brief A Disjoint Set Union (DSU) or Union-Find data structure.
 * Used to efficiently track the connected components of a graph.
 */
struct DSU {
    std::vector<int> parent;
    int components;

    // Constructor initializes n vertices as n separate components.
    DSU(int n) {
        parent.resize(n);
        std::iota(parent.begin(), parent.end(), 0); // Fills parent with 0, 1, ..., n-1
        components = n;
    }

    // Find the representative (root) of the set containing element i, with path compression.
    int find(int i) {
        if (parent[i] == i) {
            return i;
        }
        return parent[i] = find(parent[i]);
    }

    // Unite the sets containing elements i and j.
    void unite(int i, int j) {
        int root_i = find(i);
        int root_j = find(j);
        if (root_i != root_j) {
            parent[root_i] = root_j;
            components--; // A merge reduces the number of components by one.
        }
    }
};

int main() {
    // Optimize standard input/output operations
    std::ios_base::sync_with_stdio(false);
    std::cin.tie(NULL);

    int n, m, t;
    std::cin >> n >> m >> t;

    // Read the color values first
    std::vector<long long> a_values(t);
    for (int i = 0; i < t; ++i) {
        std::cin >> a_values[i];
    }

    // Then, read the edges
    std::vector<std::pair<int, int>> edges(m);
    for (int i = 0; i < m; ++i) {
        std::cin >> edges[i].first >> edges[i].second;
    }

    // coeffs[j] will store the coefficient for the k^j term in the chromatic polynomial.
    std::vector<long long> coeffs(n + 1, 0);

    // Iterate through all 2^m subsets of edges using a bitmask.
    for (int i = 0; i < (1 << m); ++i) {
        DSU dsu(n);
        int num_edges_in_subset = 0;

        // For the current subset (represented by the mask i), build the graph.
        for (int j = 0; j < m; ++j) {
            if ((i >> j) & 1) { // If the j-th edge is in the subset
                num_edges_in_subset++;
                dsu.unite(edges[j].first, edges[j].second);
            }
        }

        // The term in the inclusion-exclusion formula is (-1)^|S| * k^(c(S)).
        // We add (-1)^|S| to the coefficient of k^(c(S)).
        long long term = (num_edges_in_subset % 2 == 1) ? -1 : 1;

        coeffs[dsu.components] = (coeffs[dsu.components] + term + MOD) % MOD;
    }

    // Process each query using the stored a_values
    for (long long a: a_values) {
        // The number of colors must be non-negative
        long long k = (a % MOD + MOD) % MOD;

        long long total_colorings = 0;
        if (n > 0) {
            // Evaluate the polynomial P(G, k) = C_n*k^n + ... + C_1*k using Horner's method.
            total_colorings = coeffs[n];
            for (int j = n - 1; j >= 1; --j) {
                total_colorings = (total_colorings * k + coeffs[j]) % MOD;
            }
            // Since the polynomial has no constant term (C_0 = 0), we need one final
            // multiplication by k.
            total_colorings = (total_colorings * k) % MOD;
        }

        // Ensure the result is positive
        total_colorings = (total_colorings + MOD) % MOD;
        std::cout << total_colorings << std::endl;
    }

    return 0;
}
