#!/usr/bin/env python3
"""
Integer‑linear‑programme replacement for the task in the prompt.

*  O(D∙N²)   time
*  O(N²)     memory
*  Pure Python – no external dependencies
"""

import sys
from itertools import combinations


# ────────── utilities ────────── #

def line() -> str:
    return sys.stdin.readline().strip()


def line_ints() -> list[int]:
    return list(map(int, line().split()))


def int_median(values):
    """Any integer median minimises Σ|x−m| for integer x."""
    s = sorted(values)
    return s[(len(s) - 1) // 2]


# ────────── core helpers ────────── #

def distances(A):
    """Full pairwise Manhattan distances of the matrix A (N×D)."""
    N = len(A)
    dist = [[0] * N for _ in range(N)]
    for i, k in combinations(range(N), 2):
        d = sum(abs(A[i][j] - A[k][j]) for j in range(len(A[0])))
        dist[i][k] = dist[k][i] = d
    return dist


def embed_on_line(dist):
    """
    Test whether the metric 'dist' can be realised on the real line
    and, if so, return coordinates f[0…N‑1] with f[a]=0, f[b]=M
    where (a,b) is a diameter of the metric.

    Criterion:   pick a,b with maximal d(a,b)=M.
                 Metric is line‑embeddable  ⇔
                 ∀v,  d(a,v)+d(v,b) == M.
    """
    N = len(dist)

    # 1) pick any diametral pair
    a, b, M = 0, 0, -1
    for i, k in combinations(range(N), 2):
        if dist[i][k] > M:
            a, b, M = i, k, dist[i][k]

    # 2) check the line condition
    for v in range(N):
        if dist[a][v] + dist[v][b] != M:
            return None   # not a path metric

    # 3) coordinates: f(v) = d(a,v)
    f = [dist[a][v] for v in range(N)]
    return f, M


def solve_instance(N, D, A):
    dist = distances(A)

    embed = embed_on_line(dist)
    if embed is None:
        return None, None
    f, _ = embed          # base coordinates (non‑negative)

    # medians of the “frozen” columns are optimal no matter which p
    col_medians = [int_median([A[i][q] for i in range(N)]) for q in range(D)]

    best_obj = None
    best_X = None

    for p in range(D):                 # try every free column
        for sign in (+1, -1):          # two orientations of the line
            g = [sign * x for x in f]

            # best integer shift in column p
            shift = int_median([A[i][p] - g[i] for i in range(N)])
            Xp = [g[i] + shift for i in range(N)]

            # objective contribution from column p
            obj = sum(abs(Xp[i] - A[i][p]) for i in range(N))

            # add contributions from the frozen columns
            for q in range(D):
                if q == p:
                    continue
                c = col_medians[q]
                obj += sum(abs(c - A[i][q]) for i in range(N))

            if best_obj is None or obj < best_obj:
                best_obj = obj
                # assemble full matrix
                X = []
                for i in range(N):
                    row = []
                    for q in range(D):
                        row.append(Xp[i] if q == p else col_medians[q])
                    X.append(row)
                best_X = X

    return best_obj, best_X


# ────────── driver ────────── #

def main():
    N, D = line_ints()
    A = [line_ints() for _ in range(N)]

    obj, X = solve_instance(N, D, A)

    if X is None:
      #print("NO SOLUTION")
        print(-1)
        return

    #for row in X:
        #print(" ".join(map(str, row)))
    print(obj)


if __name__ == "__main__":
    main()
