#include <iostream>
#include <vector>
#include <algorithm>
#include <climits>
#include <map>

using namespace std;

int N, D;
vector<vector<long long>> crows;
map<pair<int,int>, long long> distCache;

long long getDist(int i, int j) {
    if (i > j) swap(i, j);
    auto key = make_pair(i, j);
    if (distCache.count(key)) return distCache[key];

    long long d = 0;
    for (int k = 0; k < D; k++) {
        d += abs(crows[i][k] - crows[j][k]);
    }
    return distCache[key] = d;
}

long long computeCost(const vector<long long>& targetPos, int alignDim) {
    // Find optimal offset for alignment dimension
    vector<long long> offsets;
    for (int i = 0; i < N; i++) {
        offsets.push_back(crows[i][alignDim] - targetPos[i]);
    }
    sort(offsets.begin(), offsets.end());
    long long optOffset = offsets[N / 2];

    long long cost = 0;

    // Cost for alignment dimension
    for (int i = 0; i < N; i++) {
        cost += abs((targetPos[i] + optOffset) - crows[i][alignDim]);
    }

    // Cost for other dimensions (precompute once)
    static vector<long long> otherDimsCost;
    if (otherDimsCost.empty()) {
        otherDimsCost.resize(D, 0);
        for (int dim = 0; dim < D; dim++) {
            vector<long long> coords;
            for (int i = 0; i < N; i++) {
                coords.push_back(crows[i][dim]);
            }
            sort(coords.begin(), coords.end());
            long long median = coords[N / 2];

            for (int i = 0; i < N; i++) {
                otherDimsCost[dim] += abs(crows[i][dim] - median);
            }
        }
    }

    // Add cost from all dimensions except alignDim
    for (int dim = 0; dim < D; dim++) {
        if (dim != alignDim) {
            cost += otherDimsCost[dim];
        }
    }

    return cost;
}

long long solve() {
    long long minCost = LLONG_MAX;
    bool foundEmbedding = false;

    // Try different pairs as reference points
    for (int ref1 = 0; ref1 < min(N, 10); ref1++) {
        for (int ref2 = ref1 + 1; ref2 < min(N, 10); ref2++) {
            // Try both orientations
            for (int orientation = 0; orientation < 2; orientation++) {
                vector<long long> candidate(N, LLONG_MAX);
                candidate[ref1] = 0;
                candidate[ref2] = (orientation == 0) ? getDist(ref1, ref2) : -getDist(ref1, ref2);

                bool valid = true;

                // Place all other crows
                for (int i = 0; i < N; i++) {
                    if (i == ref1 || i == ref2) continue;

                    long long d1 = getDist(ref1, i);
                    long long d2 = getDist(ref2, i);
                    long long sep = candidate[ref2];

                    long long p1 = d1;
                    long long p2 = -d1;

                    if (abs(p1 - sep) == d2) {
                        candidate[i] = p1;
                    } else if (abs(p2 - sep) == d2) {
                        candidate[i] = p2;
                    } else {
                        valid = false;
                        break;
                    }
                }

                if (!valid) continue;

                // Verify all pairwise distances (sample for large N)
                bool allCorrect = true;
                int checkLimit = min(N, 100);
                for (int i = 0; i < checkLimit && allCorrect; i++) {
                    for (int j = i + 1; j < checkLimit; j++) {
                        if (abs(candidate[i] - candidate[j]) != getDist(i, j)) {
                            allCorrect = false;
                            break;
                        }
                    }
                }

                if (!allCorrect) continue;

                // Found a valid embedding! Try aligning along each dimension
                foundEmbedding = true;
                for (int dim = 0; dim < D; dim++) {
                    long long cost = computeCost(candidate, dim);
                    minCost = min(minCost, cost);
                }
            }
        }
    }

    return foundEmbedding ? minCost : -1;
}

int main() {
    ios_base::sync_with_stdio(false);
    cin.tie(nullptr);

    cin >> N >> D;

    crows.assign(N, vector<long long>(D));
    for (int i = 0; i < N; i++) {
        for (int j = 0; j < D; j++) {
            cin >> crows[i][j];
        }
    }

    if (N == 1) {
        cout << 0 << endl;
        return 0;
    }

    long long answer = solve();
    cout << answer << endl;

    return 0;
}
