#include <iostream>
#include <vector>
#include <numeric>
#include <algorithm>
#include <cmath>
#include <string>

// Helper function to print __int128_t, as it's not natively supported by iostream.
std::ostream& operator<<(std::ostream& os, __int128_t value) {
    if (value == 0) return os << "0";
    if (value < 0) {
        os << "-";
        value = -value;
    }
    std::string s;
    while (value > 0) {
        s += (value % 10) + '0';
        value /= 10;
    }
    std::reverse(s.begin(), s.end());
    return os << s;
}

// Function to calculate the Manhattan distance between two points.
long long manhattan_distance(const std::vector<long long>& p1, const std::vector<long long>& p2) {
    long long dist = 0;
    for (size_t i = 0; i < p1.size(); ++i) {
        dist += std::abs(p1[i] - p2[i]);
    }
    return dist;
}

// Function to calculate the minimum cost to make all elements in a vector equal.
// The optimal point to converge to is the median.
__int128_t get_median_cost(std::vector<long long>& nums) {
    if (nums.empty()) {
        return 0;
    }
    size_t n = nums.size();
    std::sort(nums.begin(), nums.end());
    long long median = nums[n / 2];
    
    __int128_t cost = 0;
    for (long long num : nums) {
        cost += std::abs(num - median);
    }
    return cost;
}

int main() {
    // Fast I/O
    std::ios_base::sync_with_stdio(false);
    std::cin.tie(NULL);

    int n, d;
    std::cin >> n >> d;

    std::vector<std::vector<long long>> points(n, std::vector<long long>(d));
    for (int i = 0; i < n; ++i) {
        for (int j = 0; j < d; ++j) {
            std::cin >> points[i][j];
        }
    }
    
    if (n <= 2) {
        // With 2 or fewer points, a solution always exists and cost can be calculated
        // based on the first interpretation, but for this problem, we can simplify.
        // The logic below handles it, but this is a base case.
    }

    // --- Feasibility Check (Literal Interpretation) ---
    // First, check if it's possible to place points on a line while preserving
    // the original Manhattan distances. This requires finding the two farthest points
    // and verifying all other points lie on the segment between them.
    
    int farthest_a = 0, farthest_b = 1;
    long long max_dist = 0;

    if (n > 1) {
        // This O(N^2*D) approach might be slow if N is large, but is necessary
        // for the literal interpretation of the problem.
        for (int i = 0; i < n; ++i) {
            for (int j = i + 1; j < n; ++j) {
                long long dist = manhattan_distance(points[i], points[j]);
                if (dist > max_dist) {
                    max_dist = dist;
                    farthest_a = i;
                    farthest_b = j;
                }
            }
        }
    }


    bool possible = true;
    if (n > 2) {
        for (int i = 0; i < n; ++i) {
            if (i == farthest_a || i == farthest_b) continue;
            if (manhattan_distance(points[farthest_a], points[i]) + manhattan_distance(points[i], points[farthest_b]) != max_dist) {
                possible = false;
                break;
            }
        }
    }

    if (!possible) {
        std::cout << -1 << std::endl;
        return 0;
    }

    // --- Cost Calculation (if a solution exists) ---
    
    // Target coordinates for the varying dimension, relative to point farthest_a
    std::vector<long long> q(n);
    if (n > 1) {
        for(int i = 0; i < n; ++i) {
            q[i] = manhattan_distance(points[farthest_a], points[i]);
        }
    }

    // Pre-calculate the cost to make each dimension constant
    std::vector<__int128_t> constant_dim_costs(d);
    for (int j = 0; j < d; ++j) {
        std::vector<long long> dim_coords(n);
        for (int i = 0; i < n; ++i) {
            dim_coords[i] = points[i][j];
        }
        constant_dim_costs[j] = get_median_cost(dim_coords);
    }

    __int128_t total_constant_cost = 0;
    for(const auto& cost : constant_dim_costs) {
        total_constant_cost += cost;
    }

    __int128_t min_ops = -1;

    // Iterate through each dimension, considering it as the one that varies
    for (int k = 0; k < d; ++k) {
        __int128_t current_constant_cost = total_constant_cost - constant_dim_costs[k];
        
        // Cost for the varying dimension k, mapping to target coordinates q_i
        // Case A: target is q_i + C. Cost is median cost of (p_ik - q_i)
        std::vector<long long> case_a_values(n);
        for (int i = 0; i < n; ++i) {
            case_a_values[i] = points[i][k] - q[i];
        }
        __int128_t cost_case_a = get_median_cost(case_a_values);

        // Case B: target is -q_i + C. Cost is median cost of (p_ik + q_i)
        std::vector<long long> case_b_values(n);
        for (int i = 0; i < n; ++i) {
            case_b_values[i] = points[i][k] + q[i];
        }
        __int128_t cost_case_b = get_median_cost(case_b_values);
        
        __int128_t varying_dim_cost = std::min(cost_case_a, cost_case_b);
        __int128_t current_total_ops = current_constant_cost + varying_dim_cost;

        if (min_ops == -1 || current_total_ops < min_ops) {
            min_ops = current_total_ops;
        }
    }

    std::cout << (min_ops == -1 ? 0 : min_ops) << std::endl;

    return 0;
}
