#include <iostream>
#include <vector>
#include <algorithm>

using namespace std;

int N, M;
vector<int> initial, final_state;
int total_houses;

// Convert row, col, type to house index
// type: 0 = horizontal, 1 = vertical
int get_index(int row, int col, int type) {
    int idx = 0;
    for (int r = 0; r < row; r++) {
        idx += M + (M + 1);  // Each row has M horizontal + M+1 vertical houses
    }
    if (type == 0) {
        idx += col;
    } else {
        idx += M + col;
    }
    return idx;
}

// Gaussian elimination over GF(2)
bool can_solve(vector<vector<int>>& operations, vector<int>& target) {
    int n_ops = operations.size();
    int n_vars = total_houses;

    // Augmented matrix
    vector<vector<int>> matrix(n_ops, vector<int>(n_vars + 1));
    for (int i = 0; i < n_ops; i++) {
        for (int j = 0; j < n_vars; j++) {
            matrix[i][j] = operations[i][j];
        }
        matrix[i][n_vars] = 0;  // RHS initially 0
    }

    // Add target as constraint
    vector<int> target_row(n_vars + 1);
    for (int i = 0; i < n_vars; i++) {
        target_row[i] = 0;
    }
    target_row[n_vars] = 1;  // RHS = 1

    // We want to check if target is in span of operations
    // This is equivalent to checking if the system has a solution

    // Actually, we need to check if target can be expressed as sum of operations
    // Let's use a different approach: add target to the matrix and check rank

    // Create matrix with operations
    vector<vector<int>> mat = operations;

    // Gaussian elimination to find basis
    int rank = 0;
    vector<bool> pivot_col(n_vars, false);

    for (int col = 0; col < n_vars && rank < n_ops; col++) {
        // Find pivot
        int pivot_row = -1;
        for (int row = rank; row < n_ops; row++) {
            if (mat[row][col] == 1) {
                pivot_row = row;
                break;
            }
        }

        if (pivot_row == -1) continue;

        // Swap rows
        swap(mat[rank], mat[pivot_row]);
        pivot_col[col] = true;

        // Eliminate
        for (int row = 0; row < n_ops; row++) {
            if (row != rank && mat[row][col] == 1) {
                for (int c = 0; c < n_vars; c++) {
                    mat[row][c] ^= mat[rank][c];
                }
            }
        }

        rank++;
    }

    // Now check if target is in span
    // Try to express target as combination of basis vectors
    vector<int> result = target;

    for (int r = 0; r < rank; r++) {
        // Find pivot column
        int pcol = -1;
        for (int c = 0; c < n_vars; c++) {
            if (mat[r][c] == 1) {
                pcol = c;
                break;
            }
        }

        if (pcol != -1 && result[pcol] == 1) {
            // XOR this row
            for (int c = 0; c < n_vars; c++) {
                result[c] ^= mat[r][c];
            }
        }
    }

    // Check if result is all zeros
    for (int i = 0; i < n_vars; i++) {
        if (result[i] != 0) return false;
    }

    return true;
}

int main() {
    cin >> N >> M;

    // Calculate total number of houses
    total_houses = (2 * N + 1) * M + (2 * N + 1);
    // Wait, let me recalculate
    // We have 2N+1 rows in input
    // Odd rows (0-indexed): M houses
    // Even rows: M+1 houses
    // Total rows: 2N+1
    // Odd rows: N+1 (rows 0, 2, 4, ..., 2N)
    // Even rows: N (rows 1, 3, 5, ..., 2N-1)
    total_houses = (N + 1) * M + N * (M + 1);

    initial.resize(total_houses);
    final_state.resize(total_houses);

    // Read initial state
    int idx = 0;
    for (int i = 0; i < 2 * N + 1; i++) {
        int count = (i % 2 == 0) ? M : (M + 1);
        for (int j = 0; j < count; j++) {
            cin >> initial[idx++];
        }
    }

    // Read final state
    idx = 0;
    for (int i = 0; i < 2 * N + 1; i++) {
        int count = (i % 2 == 0) ? M : (M + 1);
        for (int j = 0; j < count; j++) {
            cin >> final_state[idx++];
        }
    }

    // Compute difference
    vector<int> diff(total_houses);
    for (int i = 0; i < total_houses; i++) {
        diff[i] = initial[i] ^ final_state[i];
    }

    // Generate operations
    vector<vector<int>> operations;

    // Horizontal lines (go through all columns)
    for (int row = 0; row <= N; row++) {
        vector<int> op(total_houses, 0);
        int idx = row * (M + M + 1);
        for (int col = 0; col < M; col++) {
            op[idx + col] = 1;
        }
        operations.push_back(op);
    }

    // Vertical lines (go through all rows)
    for (int col = 0; col <= M; col++) {
        vector<int> op(total_houses, 0);
        for (int row = 0; row < N; row++) {
            int idx = row * (M + M + 1) + M + col;
            op[idx] = 1;
        }
        operations.push_back(op);
    }

    // Perimeters of rectangular regions (all possible rectangles)
    for (int r1 = 0; r1 < N; r1++) {
        for (int c1 = 0; c1 < M; c1++) {
            for (int r2 = r1; r2 < N; r2++) {
                for (int c2 = c1; c2 < M; c2++) {
                    vector<int> op(total_houses, 0);

                    // Top and bottom edges
                    for (int c = c1; c <= c2; c++) {
                        int top_idx = r1 * (M + M + 1) + c;
                        int bottom_idx = (r2 + 1) * (M + M + 1) + c;
                        op[top_idx] = 1;
                        op[bottom_idx] = 1;
                    }

                    // Left and right edges
                    for (int r = r1; r <= r2; r++) {
                        int left_idx = r * (M + M + 1) + M + c1;
                        int right_idx = r * (M + M + 1) + M + c2 + 1;
                        op[left_idx] = 1;
                        op[right_idx] = 1;
                    }

                    operations.push_back(op);
                }
            }
        }
    }

    if (can_solve(operations, diff)) {
        cout << "Yes\n";
    } else {
        cout << "No\n";
    }

    return 0;
}
