#include <iostream>
#include <vector>
#include <cmath>
#include <numeric>
#include <set>
#include <random>
#include <map>
#include <chrono>

// Use __int128_t for coordinate calculations to prevent overflow, as intermediate
// products can exceed the capacity of a 64-bit long long.
using int128 = __int128_t;

struct Point {
    long long x, y;

    // Operator for using Point in std::set and std::map
    bool operator<(const Point& other) const {
        if (x != other.x) {
            return x < other.x;
        }
        return y < other.y;
    }
};

// Global variables for easy access within the verification function
int N;
double P;
std::vector<Point> points;
std::set<Point> point_set;
std::map<Point, int> point_to_idx;

/**
 * @brief Verifies if a given line satisfies the symmetry condition.
 * @param A, B, C The coefficients of the line Ax + By + C = 0.
 * @return True if the line is a valid axis of symmetry, false otherwise.
 */
bool verify_line(int128 A, int128 B, int128 C) {
    int128 D = A * A + B * B;
    if (D == 0) {
        return false; // Degenerate line (A=0, B=0)
    }

    int on_line_count = 0;
    int symmetric_count = 0;
    std::vector<bool> visited(N, false);

    // First pass: identify points that lie exactly on the line
    for (int i = 0; i < N; ++i) {
        if (A * points[i].x + B * points[i].y + C == 0) {
            on_line_count++;
            visited[i] = true;
        }
    }

    // Second pass: check for symmetric pairs among points not on the line
    for (int i = 0; i < N; ++i) {
        if (visited[i]) {
            continue;
        }

        Point p1 = points[i];

        // Calculate the reflection of p1 across the line Ax + By + C = 0.
        // The reflection formulas are manipulated to use only integer arithmetic,
        // preventing floating-point precision errors. This requires 128-bit integers.
        int128 num_x = (int128)p1.x * (B * B - A * A) - (int128)2 * A * B * p1.y - (int128)2 * A * C;
        int128 num_y = (int128)p1.y * (A * A - B * B) - (int128)2 * A * B * p1.x - (int128)2 * B * C;

        // If the numerators are not divisible by the denominator D, the reflection
        // is not an integer point and thus cannot be in our input set.
        if (num_x % D != 0 || num_y % D != 0) {
            visited[i] = true;
            continue;
        }

        Point reflection = {(long long)(num_x / D), (long long)(num_y / D)};

        // Check if the reflected point exists in the input set
        if (point_set.count(reflection)) {
            int reflection_idx = point_to_idx[reflection];
            // Ensure we haven't already processed this pair from the other side
            if (!visited[reflection_idx]) {
                symmetric_count += 2;
                visited[i] = true;
                visited[reflection_idx] = true;
            } else {
                visited[i] = true;
            }
        } else {
            visited[i] = true;
        }
    }

    int non_line_points = N - on_line_count;
    if (non_line_points == 0) {
        // If all points are on the line, there are no points to reflect.
        // The condition can be considered met only if the required percentage is 0.
        return P == 0;
    }

    // The condition is that the number of symmetric points is at least P percent
    // of the points *not* on the line.
    return symmetric_count * 100.0 >= non_line_points * P;
}

int main() {
    std::ios_base::sync_with_stdio(false);
    std::cin.tie(NULL);

    std::cin >> N >> P;
    points.resize(N);
    for (int i = 0; i < N; ++i) {
        std::cin >> points[i].x >> points[i].y;
        point_set.insert(points[i]);
        point_to_idx[points[i]] = i;
    }

    // Based on user feedback, it's inferred that a meaningful symmetry
    // requires at least two points to form a pair.
    // Therefore, if N < 2, no such symmetry can exist.
    if (N < 2) {
        std::cout << "NO\n";
        return 0;
    }

    if (P == 0) {
        std::cout << "YES\n";
        return 0;
    }

    // Optimization: If P is small enough, any pair of points can satisfy the condition.
    // For a generic pair, we get 2 symmetric points and 0 on the line.
    // The condition becomes: 2 >= (N - 0) * P / 100.0
    if (200.0 >= (double)N * P) {
        std::cout << "YES\n";
        return 0;
    }

    // Setup for random number generation
    std::mt19937 rng(std::chrono::steady_clock::now().time_since_epoch().count());
    std::uniform_int_distribution<int> dist(0, N - 1);

    // Number of random trials. This value can be tuned for a balance
    // between probability of correctness and execution time.
    int iterations = 400;

    for (int k = 0; k < iterations; ++k) {
        int i = dist(rng);
        int j = dist(rng);
        if (i == j) {
            j = (j + 1) % N; // Ensure two distinct points
        }

        Point p1 = points[i];
        Point p2 = points[j];

        // Candidate 1: Test the perpendicular bisector of p1 and p2
        int128 A1 = (int128)2 * (p2.x - p1.x);
        int128 B1 = (int128)2 * (p2.y - p1.y);
        int128 C1 = (int128)p1.x * p1.x + (int128)p1.y * p1.y - (int128)p2.x * p2.x - (int128)p2.y * p2.y;
        if (verify_line(A1, B1, C1)) {
            std::cout << "YES\n";
            return 0;
        }

        // Candidate 2: Test the line passing through p1 and p2
        int128 A2 = (int128)p2.y - p1.y;
        int128 B2 = (int128)p1.x - p2.x;
        int128 C2 = -A2 * p1.x - B2 * p1.y;
        if (verify_line(A2, B2, C2)) {
            std::cout << "YES\n";
            return 0;
        }
    }

    std::cout << "NO\n";

    return 0;
}
