#include <iostream>
#include <vector>
#include <algorithm>
#include <map>
#include <cmath>
#include <numeric>
#include <tuple>
#include <cstdlib> // For std::abs

using namespace std;

// Define long long to handle coordinates up to 10^6 and intermediate calculations up to 10^12.
typedef long long ll;

// Structure to represent a wagon's position.
struct Point {
    ll x, y;
};

// Function to calculate the Greatest Common Divisor (GCD) of two numbers.
// We ensure it works correctly with potentially negative inputs by taking the absolute value.
ll calculate_gcd(ll a, ll b) {
    // The Euclidean algorithm works on non-negative integers.
    a = abs(a);
    b = abs(b);
    while (b) {
        a %= b;
        swap(a, b);
    }
    return a;
    // Alternatively, if C++17 is available, std::gcd(a, b) can be used directly as it handles signs correctly.
}

// The key represents a unique line (the false escape route).
// We represent the line Ax + By = C'/2 using a tuple (A, B, C_prime),
// where (A, B) is the normalized normal vector and C_prime = 2C'.
typedef tuple<ll, ll, ll> LineKey;

void solve() {
    // Optimization for input/output operations.
    ios_base::sync_with_stdio(false);
    cin.tie(NULL);

    int N;
    int P;
    // Read N (number of wagons) and P (required percentage).
    if (!(cin >> N >> P)) return;

    // Handle the edge case N=0.
    if (N == 0) {
        // If N=0, there are 0 powerful wagons. The condition 0 >= 0*P/100 is always true.
        cout << "YES\n";
        return;
    }

    vector<Point> wagons(N);
    // Read the coordinates.
    for (int i = 0; i < N; ++i) {
        if (!(cin >> wagons[i].x >> wagons[i].y)) return;
    }

    // Map to store the counts of pairs for each unique perpendicular bisector.
    map<LineKey, int> bisector_counts;

    // Iterate over all unordered pairs of wagons (O(N^2)).
    for (int i = 0; i < N; ++i) {
        for (int j = i + 1; j < N; ++j) {
            Point P1 = wagons[i];
            Point P2 = wagons[j];

            // The vector P1P2 (dx, dy) is normal to the perpendicular bisector.
            ll dx = P2.x - P1.x;
            ll dy = P2.y - P1.y;

            // Normalize the normal vector by dividing by gcd(|dx|, |dy|).
            // Since points are distinct, dx and dy are not both zero, so the gcd is positive.
            ll g = calculate_gcd(dx, dy);

            ll A = dx / g;
            ll B = dy / g;

            // Calculate the constant term C_prime.
            // The bisector passes through the midpoint M.
            // A*Mx + B*My = C'.
            // C_prime = 2*C' = A*(x1+x2) + B*(y1+y2).
            // This calculation fits within long long (up to approx 8*10^12).
            ll C_prime = A * (P1.x + P2.x) + B * (P1.y + P2.y);

            // Canonicalize the representation to uniquely identify the line.
            // We ensure the normal vector points in a consistent direction.
            // Rule: (A > 0) or (A == 0 and B > 0).
            if (A < 0 || (A == 0 && B < 0)) {
                A = -A;
                B = -B;
                // If we flip the normal vector, we must also flip C_prime
                // to maintain the equation (-A)x + (-B)y = (-C_prime)/2.
                C_prime = -C_prime;
            }

            LineKey key = make_tuple(A, B, C_prime);
            bisector_counts[key]++;
        }
    }

    ll max_powerful_wagons = 0;

    // Find the maximum number of powerful wagons associated with any single escape route.
    for (const auto& pair : bisector_counts) {
        // K is the number of pairs sharing this bisector.
        int K = pair.second;
        // Since all input points are distinct, these K distinct pairs involve exactly 2K distinct wagons.
        // None of these wagons are on the line (as they form a pair with a distinct reflection).
        if (2LL * K > max_powerful_wagons) {
            max_powerful_wagons = 2LL * K;
        }
    }

    // Check the condition: at least P percent of the wagons must be powerful.
    // max_powerful_wagons / N >= P / 100.
    // We use integer arithmetic (cross-multiplication) to avoid precision issues.
    // We must ensure multiplication is done using long long.
    if (max_powerful_wagons * 100 >= (ll)N * P) {
        cout << "YES\n";
    } else {
        cout << "NO\n";
    }
}

int main() {
    solve();
    return 0;
}