#include <iostream>
#include <vector>

// Define the modulus for all calculations
const int MOD = 1000000007;

// Vector to store pre-computed factorials
std::vector<long long> fact;

/**
 * @brief Calculates (base^exp) % MOD using modular exponentiation.
 * This is an efficient way to compute large powers under a modulus.
 * @param base The base of the power.
 * @param exp The exponent.
 * @return The result of (base^exp) % MOD.
 */
long long power(long long base, long long exp) {
    long long res = 1;
    base %= MOD;
    while (exp > 0) {
        if (exp % 2 == 1) res = (res * base) % MOD;
        base = (base * base) % MOD;
        exp /= 2;
    }
    return res;
}

/**
 * @brief Calculates the modular multiplicative inverse of a number n.
 * It uses Fermat's Little Theorem, which states that n^(p-2) is the
 * modular inverse of n under a prime modulus p.
 * @param n The number to find the inverse of.
 * @return The modular inverse of n modulo MOD.
 */
long long modInverse(long long n) {
    return power(n, MOD - 2);
}

/**
 * @brief Pre-computes factorials up to a maximum value.
 * This avoids re-calculating factorials repeatedly.
 * @param max_val The maximum number to compute factorials for.
 */
void precompute_factorials(int max_val) {
    fact.resize(max_val + 1);
    fact[0] = 1;
    for (int i = 1; i <= max_val; i++) {
        fact[i] = (fact[i - 1] * i) % MOD;
    }
}

/**
 * @brief Calculates combinations C(n, k) modulo MOD.
 * Uses the formula C(n, k) = n! / (k! * (n-k)!).
 * Requires pre-computed factorials.
 * @param n The total number of items.
 * @param k The number of items to choose.
 * @return The number of combinations C(n, k) % MOD. Returns 0 if k > n or k < 0.
 */
long long combinations(int n, int k) {
    if (k < 0 || k > n) {
        return 0;
    }
    // Calculate n! / (k! * (n-k)!) mod MOD
    long long numerator = fact[n];
    long long denominator = (fact[k] * fact[n - k]) % MOD;
    return (numerator * modInverse(denominator)) % MOD;
}

int main() {
    // Optimize standard input/output operations
    std::ios_base::sync_with_stdio(false);
    std::cin.tie(NULL);

    int n;
    std::cin >> n;

    // The maximum value for which we need a factorial is 4*n.
    precompute_factorials(4 * n);

    long long total_ways = 0;

    // The sum runs from k=1 up to n/2, because for k > n/2,
    // the term C(4n - 4k, 2n) becomes zero (since 4n - 4k < 2n).
    for (int k = 1; k <= n / 2; ++k) {
        // Calculate the two combination terms for the current k
        long long comb_n_k = combinations(n, k);
        long long comb_remaining = combinations(4 * n - 4 * k, 2 * n);

        // Combine them to get the full term for this k
        long long term = (comb_n_k * comb_remaining) % MOD;

        // Apply the (-1)^(k-1) factor from the Inclusion-Exclusion Principle
        if (k % 2 == 1) {
            // If k is odd, (-1)^(k-1) is positive, so we add the term
            total_ways = (total_ways + term) % MOD;
        } else {
            // If k is even, (-1)^(k-1) is negative, so we subtract the term
            // Add MOD before taking the modulus to handle negative results
            total_ways = (total_ways - term + MOD) % MOD;
        }
    }

    std::cout << total_ways << std::endl;

    return 0;
}

