#include <iostream>
#include <string>
#include <vector>
#include <list>
#include <utility> // For std::move
#include <numeric> // For std::accumulate

/**
 * @brief This version uses a linked list of string chunks and hashing to solve the problem.
 *
 * The core idea is to represent the string as a `std::list<std::string>`.
 * This allows for faster removal of substrings without shifting large amounts of data,
 * which is the main drawback of using a single `std::string`. Hashing (Rabin-Karp)
 * is used to speed up the search for the query string.
 *
 * CAVEAT: The logic for finding matches that span across two different chunks
 * and then modifying the list accordingly is highly complex. This implementation
 * simplifies the problem by temporarily "flattening" the list into a single string
 * to perform the search with hashing. It then rebuilds the string and stores it
 * as a new single chunk in the list. This uses the requested concepts but doesn't
 * fully realize the theoretical performance benefits of a true Rope data structure,
 * which would avoid the flattening step altogether.
 */
void solve() {
    // Fast I/O
    std::ios_base::sync_with_stdio(false);
    std::cin.tie(NULL);

    int n_initial, q;
    std::cin >> n_initial >> q;

    std::string s_initial;
    std::cin >> s_initial;

    // The string is represented as a list of chunks. Initially, one chunk.
    std::list<std::string> s_chunks;
    if (!s_initial.empty()) {
        s_chunks.push_back(s_initial);
    }

    // Precompute powers of p for hashing. Max length of string is 10^5.
    long long p = 31;
    long long m = 1e9 + 9;
    std::vector<long long> p_pow(100001);
    p_pow[0] = 1;
    for (size_t i = 1; i < p_pow.size(); ++i) {
        p_pow[i] = (p_pow[i - 1] * p) % m;
    }

    for (int i = 0; i < q; ++i) {
        int ni;
        std::cin >> ni;
        std::string si;
        std::cin >> si;

        if (si.empty() || s_chunks.empty()) {
            std::cout << 0 << std::endl;
            continue;
        }

        // --- Step 1: Flatten the list into a single string for searching ---
        // This is a concession for simplicity. A fully optimal solution would
        // search the list structure directly, which is significantly more complex.
        std::string current_s;
        size_t total_len = 0;
        for(const auto& chunk : s_chunks) {
            total_len += chunk.length();
        }
        current_s.reserve(total_len);
        for(const auto& chunk : s_chunks) {
            current_s.append(chunk);
        }

        if (current_s.length() < si.length()) {
            std::cout << 0 << std::endl;
            continue;
        }

        // --- Step 2: Find matches using Hashing (Rabin-Karp) ---
        // Calculate the hash of the query string
        long long si_hash = 0;
        for (size_t j = 0; j < si.length(); ++j) {
            si_hash = (si_hash + (si[j] - 'a' + 1) * p_pow[j]) % m;
        }

        // Calculate prefix hashes for the current main string
        std::vector<long long> h(current_s.length() + 1, 0);
        for (size_t j = 0; j < current_s.length(); ++j) {
            h[j + 1] = (h[j] + (current_s[j] - 'a' + 1) * p_pow[j]) % m;
        }

        std::vector<size_t> positions;
        size_t pos = 0;
        while (pos <= current_s.length() - si.length()) {
            // Calculate hash of the current window
            long long cur_h = (h[pos + si.length()] - h[pos] + m) % m;

            // Compare hashes (adjusting for powers of p)
            if (cur_h == (si_hash * p_pow[pos]) % m) {
                // Hash match found, verify with string comparison to avoid collisions
                if (current_s.substr(pos, si.length()) == si) {
                    positions.push_back(pos);
                    pos += si.length(); // Move past this match for non-overlapping
                    continue;
                }
            }
            pos++;
        }

        // --- Step 3: Output the count ---
        std::cout << positions.size() << std::endl;

        // --- Step 4: Rebuild the string and update the chunk list ---
        if (!positions.empty()) {
            std::string new_s;
            new_s.reserve(current_s.length() - positions.size() * si.length());
            size_t last_pos = 0;
            for (size_t p : positions) {
                new_s.append(current_s, last_pos, p - last_pos);
                last_pos = p + si.length();
            }
            new_s.append(current_s, last_pos, current_s.length() - last_pos);

            // Clear the old list and replace with a single new chunk.
            s_chunks.clear();
            if (!new_s.empty()) {
                s_chunks.push_back(std::move(new_s));
            }
        }
    }
}

int main() {
    solve();
    return 0;
}
